#!/usr/bin/env python3
"""
PHAISTOS DISC: A SERIOUS CRYPTOGRAPHIC ATTACK
==============================================
Treating this as what it might actually be: 
A complex human message, not a religious artifact.

What if the ancient Minoans were as clever as us?
"""

import re
from collections import Counter, defaultdict
from itertools import combinations

# The full transcription
side_a_words = [
    [2,12,13,1,18],           # A1
    [24,40,12],               # A2
    [29,45,7],                # A3
    [29,29,34],               # A4
    [2,12,4,40,33],           # A5
    [27,45,7,12],             # A6
    [27,44,8],                # A7
    [2,12,6,18],              # A8 (missing sign)
    [31,26,35],               # A9
    [2,12,41,19,35],          # A10
    [1,41,40,7],              # A11
    [2,12,32,23,38],          # A12
    [39,11],                  # A13
    [2,27,25,10,23,18],       # A14
    [28,1],                   # A15
    [2,12,31,26],             # A16
    [2,12,27,27,35,37,21],    # A17
    [33,23],                  # A18
    [2,12,31,26],             # A19
    [2,27,25,10,23,18],       # A20
    [28,1],                   # A21
    [2,12,31,26],             # A22
    [2,12,27,14,32,18,27],    # A23
    [6,18,17,19],             # A24
    [31,26,12],               # A25
    [2,12,13,1],              # A26
    [23,19,35],               # A27
    [10,3,38],                # A28
    [2,12,27,27,35,37,21],    # A29
    [13,1],                   # A30
    [10,3,38],                # A31
]

side_b_words = [
    [2,12,22,40,7],           # B1
    [27,45,7,35],             # B2
    [2,37,23,5],              # B3
    [22,25,27],               # B4
    [33,24,20,12],            # B5
    [16,23,18,43],            # B6
    [13,1,39,33],             # B7
    [15,7,13,1,18],           # B8
    [22,37,42,25],            # B9
    [7,24,40,35],             # B10
    [2,26,36,40],             # B11
    [27,25,38,1],             # B12
    [29,24,24,20,35],         # B13
    [16,14,18],               # B14
    [29,33,1],                # B15
    [6,35,32,39,33],          # B16
    [2,9,27,1],               # B17
    [29,36,7,8],              # B18
    [29,8,13],                # B19
    [29,45,7],                # B20
    [22,29,36,7,8],           # B21
    [27,34,23,25],            # B22
    [7,18,35],                # B23
    [7,45,7],                 # B24
    [7,23,18,24],             # B25
    [22,29,36,7,8],           # B26
    [9,30,39,18,7],           # B27
    [2,6,35,23,7],            # B28
    [29,34,23,25],            # B29
    [45,7],                   # B30
]

sign_names = {
    1: "PEDESTRIAN", 2: "PLUMED HEAD", 3: "TATTOOED HEAD", 4: "CAPTIVE",
    5: "CHILD", 6: "WOMAN", 7: "HELMET", 8: "GAUNTLET", 9: "TIARA",
    10: "ARROW", 11: "BOW", 12: "SHIELD", 13: "CLUB", 14: "MANACLES",
    15: "MATTOCK", 16: "SAW", 17: "LID", 18: "BOOMERANG", 19: "PAPYRUS",
    20: "DOLIUM", 21: "COMB", 22: "SLING", 23: "COLUMN", 24: "BEEHIVE",
    25: "SHIP", 26: "HORN", 27: "HIDE", 28: "BULLS LEG", 29: "CAT",
    30: "RAM", 31: "EAGLE", 32: "DOVE", 33: "TUNNY", 34: "BEE",
    35: "PLANE TREE", 36: "VINE", 37: "ROSETTE", 38: "LILY",
    39: "OX-BACK", 40: "WAVY BAND", 41: "HELMET HEAD", 42: "COIF",
    43: "STRAINER", 44: "SMALL AXE", 45: "DOUBLE-AXE"
}

print("=" * 75)
print("CRYPTOGRAPHIC ATTACK ON THE PHAISTOS DISC")
print("Hypothesis: This is a DIPLOMATIC/POLITICAL document, not religious")
print("=" * 75)

# ATTACK 1: Word Structure Analysis
print("\n" + "═" * 75)
print("ATTACK 1: WORD STRUCTURE ANALYSIS")
print("═" * 75)

all_words = side_a_words + side_b_words

# Analyze word beginnings and endings
first_signs = Counter(w[0] for w in all_words)
last_signs = Counter(w[-1] for w in all_words)

print("\nFirst sign distribution (potential prefixes/determinatives):")
for sign, count in first_signs.most_common(10):
    pct = count / len(all_words) * 100
    print(f"  {sign:02d} ({sign_names[sign]:12s}): {count:2d} words ({pct:4.1f}%)")

print("\nLast sign distribution (potential suffixes/case endings):")
for sign, count in last_signs.most_common(10):
    pct = count / len(all_words) * 100
    print(f"  {sign:02d} ({sign_names[sign]:12s}): {count:2d} words ({pct:4.1f}%)")

# ATTACK 2: Identify grammatical categories
print("\n" + "═" * 75)
print("ATTACK 2: GRAMMATICAL CATEGORY IDENTIFICATION")
print("═" * 75)

# Words starting with 02 (PLUMED HEAD)
type_02 = [w for w in all_words if w[0] == 2]
# Words starting with other signs
type_other = [w for w in all_words if w[0] != 2]

print(f"\nWords beginning with 02 (PLUMED HEAD): {len(type_02)}")
print(f"Words beginning with other signs: {len(type_other)}")
print(f"\nHYPOTHESIS: Sign 02 is a DETERMINATIVE marking a word class")
print("           Like Egyptian hieroglyphs use determinatives for categories")

# Analyze what follows 02
print("\nWhat follows sign 02:")
follows_02 = Counter(w[1] for w in type_02 if len(w) > 1)
for sign, count in follows_02.most_common():
    print(f"  02-{sign:02d}: {count:2d} times ({sign_names[sign]})")

print("\n>>> FINDING: 02 is almost always followed by 12 (SHIELD) or 27 (HIDE)")
print(">>> This suggests 02-12-X and 02-27-X are different word categories")

# ATTACK 3: Looking for proper names
print("\n" + "═" * 75)
print("ATTACK 3: IDENTIFYING POTENTIAL PROPER NAMES")
print("═" * 75)

print("""
PRINCIPLE: In ancient texts, names often:
  1. Appear at specific positions (beginning, end of clauses)
  2. Have consistent spelling across occurrences
  3. May lack certain grammatical markers

Looking for candidates...
""")

# Find exact repeated sequences
word_tuples = [tuple(w) for w in all_words]
word_counts = Counter(word_tuples)

print("Exact repeated words (potential names or titles):")
for word, count in sorted(word_counts.items(), key=lambda x: -x[1]):
    if count > 1:
        signs = "-".join(f"{s:02d}" for s in word)
        names = [sign_names[s][:4] for s in word]
        print(f"  [{signs}] appears {count}x  ({'-'.join(names)})")

# ATTACK 4: Positional Analysis
print("\n" + "═" * 75)
print("ATTACK 4: POSITIONAL ANALYSIS")
print("═" * 75)

print("""
If this is a LETTER or TREATY, we expect:
  - Opening formula (sender identification)
  - Address to recipient
  - Body with terms/message
  - Closing formula

Analyzing positional patterns...
""")

# First and last words of each side
print(f"Side A first word: {side_a_words[0]} = ", end="")
print("-".join(sign_names[s][:4] for s in side_a_words[0]))

print(f"Side A last word:  {side_a_words[-1]} = ", end="")
print("-".join(sign_names[s][:4] for s in side_a_words[-1]))

print(f"\nSide B first word: {side_b_words[0]} = ", end="")
print("-".join(sign_names[s][:4] for s in side_b_words[0]))

print(f"Side B last word:  {side_b_words[-1]} = ", end="")
print("-".join(sign_names[s][:4] for s in side_b_words[-1]))

# ATTACK 5: The Achterberg Hypothesis (Luwian)
print("\n" + "═" * 75)
print("ATTACK 5: TESTING THE LUWIAN HYPOTHESIS")
print("═" * 75)

print("""
Achterberg et al. (2004) proposed this is a letter in LUWIAN
from a western Anatolian ruler to King Nestor of Pylos.

Their proposed sign values (partial):
""")

luwian_values = {
    2: "ta",    # PLUMED HEAD
    7: "ti",    # HELMET  
    12: "ta",   # SHIELD
    27: "wa",   # HIDE
    45: "LABRYS (divine/royal)",  # DOUBLE-AXE
    1: "sa",    # PEDESTRIAN
    13: "ku",   # CLUB
    18: "na",   # BOOMERANG
    23: "ra",   # COLUMN
    25: "pi",   # SHIP
    26: "nu",   # HORN
    29: "ma",   # CAT
    31: "AQUILA (eagle symbol)",  # EAGLE
    35: "i",    # PLANE TREE
}

print("Proposed phonetic values:")
for sign, value in sorted(luwian_values.items()):
    print(f"  {sign:02d} ({sign_names[sign]:12s}) = {value}")

print("\nApplying to key sequences:")

def transliterate(word, values):
    """Attempt transliteration using proposed values"""
    result = []
    for sign in word:
        if sign in values:
            result.append(values[sign])
        else:
            result.append(f"[{sign}]")
    return "-".join(result)

key_words = [
    ("A1 (opening)", [2,12,13,1,18]),
    ("A14/A20 (refrain)", [2,27,25,10,23,18]),
    ("A16/A19/A22", [2,12,31,26]),
    ("B30 (closing)", [45,7]),
]

for name, word in key_words:
    trans = transliterate(word, luwian_values)
    print(f"  {name}: {trans}")

# ATTACK 6: Statistical Word Classification
print("\n" + "═" * 75)
print("ATTACK 6: STATISTICAL WORD CLASSIFICATION")
print("═" * 75)

print("""
HYPOTHESIS: We can identify word classes by their structure

Class A: Words starting with 02-12 (13 words)
  → Possibly NOUNS with definite article
  
Class B: Words starting with 02-27 (2 words)  
  → Possibly NOUNS of different class or VERBS
  
Class C: Words starting with 29 (CAT) (8 words on B side)
  → Possibly VERBS or a different grammatical category
  
Class D: Short words (2-3 signs)
  → Possibly PARTICLES, CONJUNCTIONS, or NAMES
""")

# Classify words
class_a = [w for w in all_words if len(w) >= 2 and w[0] == 2 and w[1] == 12]
class_b = [w for w in all_words if len(w) >= 2 and w[0] == 2 and w[1] == 27]
class_c = [w for w in all_words if w[0] == 29]
class_d = [w for w in all_words if len(w) <= 3]

print(f"Class A (02-12-...): {len(class_a)} words")
print(f"Class B (02-27-...): {len(class_b)} words")
print(f"Class C (29-...):    {len(class_c)} words")
print(f"Class D (short):     {len(class_d)} words")

# ATTACK 7: Semantic Field Hypothesis
print("\n" + "═" * 75)
print("ATTACK 7: SEMANTIC FIELD HYPOTHESIS")
print("═" * 75)

print("""
What if the PICTOGRAPHIC signs give us SEMANTIC hints?

Sign clusters that might form semantic fields:

MARITIME/TRAVEL: 25 (SHIP), 40 (WAVY BAND), 36 (VINE - trade goods?)
MILITARY: 7 (HELMET), 12 (SHIELD), 13 (CLUB), 8 (GAUNTLET)
ROYALTY: 2 (PLUMED HEAD), 45 (DOUBLE-AXE = labrys), 9 (TIARA)
AGRICULTURE: 34 (BEE), 24 (BEEHIVE), 35 (PLANE TREE), 38 (LILY)
PEOPLE: 1 (PEDESTRIAN), 6 (WOMAN), 4 (CAPTIVE), 5 (CHILD)

This distribution suggests a TREATY or TRADE AGREEMENT:
  - Maritime terms (shipping routes?)
  - Military terms (alliance?)
  - Royal authority markers
  - Agricultural goods (tribute?)
  - People (possibly hostages or marriages?)
""")

# Count semantic fields
maritime = [25, 40, 36]
military = [7, 12, 13, 8]
royalty = [2, 45, 9]
agriculture = [34, 24, 35, 38]
people = [1, 6, 4, 5]

all_signs = []
for w in all_words:
    all_signs.extend(w)

def count_field(signs, field):
    return sum(1 for s in signs if s in field)

print("\nSemantic field distribution:")
print(f"  Maritime signs:    {count_field(all_signs, maritime):3d}")
print(f"  Military signs:    {count_field(all_signs, military):3d}")
print(f"  Royalty signs:     {count_field(all_signs, royalty):3d}")
print(f"  Agriculture signs: {count_field(all_signs, agriculture):3d}")
print(f"  People signs:      {count_field(all_signs, people):3d}")

# ATTACK 8: Structural Template
print("\n" + "═" * 75)
print("ATTACK 8: FITTING A MESSAGE TEMPLATE")
print("═" * 75)

print("""
PROPOSED READING - A DIPLOMATIC LETTER:

If this is a letter from a Luwian/Anatolian ruler to a Cretan king,
it might follow this structure:

═══════════════════════════════════════════════════════════════════════
SIDE A - THE FORMAL DECLARATION
═══════════════════════════════════════════════════════════════════════

[A1] 02-12-13-01-18  
     "The [royal] KU-SA-NA" = "The King speaks" or Royal name
     
[A2-A3] Introduction/Greeting formula

[A4-A12] FIRST CLAUSE - possibly about military matters
         (Note: lots of SHIELD, HELMET signs)
         
[A13-A15] REFRAIN A - "By the ship-alliance of [place name]"
[A16] REFRAIN B - "Eagle-Horn" = possibly a divine witness formula

[A17-A19] SECOND CLAUSE - possibly about trade/tribute
          (Note: HIDE-HIDE suggests goods, COMB might be textiles)
          
[A20-A22] REFRAIN A + B REPEATED - standard treaty formula

[A23-A27] THIRD CLAUSE - possibly marriage alliance?
          (Note: WOMAN sign appears, MANACLES might mean "bound to")
          
[A28-A31] CLOSING FORMULA - "Arrow-Head-Flower" brackets
          Like "so let it be" or "sealed"

═══════════════════════════════════════════════════════════════════════
SIDE B - THE SPECIFIC TERMS
═══════════════════════════════════════════════════════════════════════

[B1-B6] Opening + identifications
        (Different pattern suggests specific content)
        
[B7-B18] Main body - the actual terms
         (Longest paragraph - lots of detail)
         CAT sign appears frequently - possibly a verb "to give/send"?
         
[B19-B26] Secondary terms or conditions

[B27-B30] Closing - ends with DOUBLE-AXE + HELMET
          = "By the sacred labrys, [signed]"
""")

# ATTACK 9: The Big Picture
print("\n" + "═" * 75)
print("ATTACK 9: SYNTHESIS - WHAT THIS DISC LIKELY SAYS")
print("═" * 75)

print("""
MOST LIKELY INTERPRETATION:

This is a TREATY or DIPLOMATIC LETTER, possibly:

1. From an Anatolian (Luwian) king to Crete
2. Concerning a military alliance
3. Including trade agreements (hides, textiles, agricultural goods)
4. Possibly sealed by a marriage alliance
5. Witnessed by divine authority (the labrys/double-axe)

The REFRAINS are not prayers - they are LEGAL FORMULAE,
like "Be it resolved" or "Witnessed by" in modern documents.

The SPIRAL is not mystical - it's PRACTICAL:
  - Side A: The public declaration (read aloud?)
  - Side B: The specific terms (for the archive?)

The STAMPS are not mysterious - they're EFFICIENT:
  - Mass-produce copies for multiple recipients
  - Ensure exact wording across copies
  - Like a Bronze Age printing press for official documents

═══════════════════════════════════════════════════════════════════════

THIS IS NOT A RELIGIOUS ARTIFACT.
THIS IS A PIECE OF BRONZE AGE BUREAUCRACY.

And bureaucracy, my friends, is the backbone of civilization.

═══════════════════════════════════════════════════════════════════════
""")

print("\n" + "═" * 75)
print("NEXT STEPS FOR A MODERN CRYPTOGRAPHER")
print("═" * 75)

print("""
To make further progress, we need:

1. COMPUTATIONAL LINGUISTICS
   - Apply n-gram analysis from Linear A/B
   - Use machine learning on Bronze Age corpus
   - Test statistical similarity to known languages

2. COMPARATIVE ANALYSIS  
   - Compare with Luwian hieroglyphs systematically
   - Cross-reference with Linear A sign values
   - Check against Anatolian treaty formats

3. ARCHAEOLOGICAL CONTEXT
   - What was Phaistos's political situation c. 1700 BCE?
   - Who were their trading partners?
   - What treaties do we know about?

4. LOCAL AI ANALYSIS
   - Run embeddings on all Aegean Bronze Age texts
   - Look for statistical fingerprints
   - Test hypothetical readings against known patterns

The tools exist. The data exists. 
We just need someone brave enough to try.
""")
